﻿/*
	VERSION:	1.2
	
	WHAT THIS IS FOR:
		This system allows you to pause/resume many objects & movieClips at once.
	
	USAGE:
		#include "loopManager.as"
		_global.LOOP = makeLoopManager();
		
		Name the loop functions:		loop()
		
		
	FUNCTIONS:
		addObject( myObject, setName )							Adds an object to a loopSet.  (and creates a new loopSet if neccessary)
		removeObject( myObject, setName )					Removes an object from a loopSet.
		removeSet( setName )										Removes a loopSet.
		pauseSet( setName, pauseValue )						Pause/resume all objects in a loopSet.
		pauseObject( myObject, pauseValue )					Pause/resume an object.
	
	HOW THIS WORKS:
		This manually calls every object's loop() function.
		Groups of objects are organized into sets, 
		allowing many objects to be paused at once, 
		via their "isPaused" variables.
		
	NOTE:
		Pause-able movieClips should not use onEnterFrame() for their loops.		
		Loop functions must be named:		loop()
		
		Pause-able objects should not use the variable "isPaused" internally, since this system uses it.
		
		Many systems have their own internal loop intervals.  
		Those intervals must be cleared first before adding those objects to this system,
		otherwise their loops will get called twice per frame.
*/
makeLoopManager = function()
{
	// create container
	var _this = new Object();
	
	_this.loopSets = new Object();		// contains sets of objects to loop.
	
	
	
	// ______________________________________________________________________________________________
	// MAIN LOOP
	_this.loop = function()
	{
		for( var L in _this.loopSets )
		{
			var thisSet = _this.loopSets[L];
			for( var obj=0; obj<thisSet.length; obj++ )
			{
				if( !thisSet[obj].isPaused )
				{
					thisSet[obj].loop();		// run this object's loop
				}// if:  This object is not paused.
			}// for:  Each object in thisSet
		}// for:  Each of the loopSets
	}// loop()
	//var FPS = 1000/30;
	var FPS = 30;
	_this.loopInterval = setInterval( _this.loop, FPS );		// 30 FPS
	
	
	
	// ______________________________________________________________________________________________
	// FUNCTIONS
	_this.addObject = function( newObject, setName )
	{
		// Create loopSet if neccessary
		if( _this.loopSets[setName] == undefined )
		{
			_this.loopSets[setName] = new Array();
		}// if:  This loopSet doesn't exist.
		var thisSet = _this.loopSets[setName];
		
		// Check whether this object is already in this loopSet
		var duplicateFound = false;
		for( var obj=0; obj<thisSet.length; obj++ )
		{
			if( thisSet[obj] == newObject )
			{
				duplicateFound = true;
			}
		}// for:  thisSet
		
		// add a reference to the newObject to this loopSet
		if( !duplicateFound )
		{
			thisSet.push( newObject );
		}// if:  No duplocate was found
	}// addObject()
	
	
	
	_this.removeObject = function( findObject, setName )
	{
		// Find the object in the specified loopSet
		var thisSet = _this.loopSets[setName];
		for( var obj=0; obj<thisSet.length; obj++ )
		{
			if( thisSet[obj] == findObject )
			{
				// remove the object
				thisSet.splice( obj, 1 );
			}// if:  object is found
		}// for:  thisSet
	}// removeObject()
	
	
	
	_this.removeSet = function( setName )
	{
		delete _this.loopSets[setName];
	}// removeSet()
	
	
	
	_this.pauseSet = function( setName, pauseValue )
	{
		var thisSet = _this.loopSets[setName];
		for( var obj in thisSet )
		{
			thisSet[obj].isPaused = pauseValue;
		}// for:  each object within this loopSet
	}// setPause()
	
	
	
	_this.pauseObject = function( myObject, pauseValue )
	{
		myObject.isPaused = pauseValue;
	}// pauseObject()
	
	
	
	// ______________________________________________________________________________________________
	// return this system's object
	return _this;
}// makeLoopManager()